const express = require("express");
const mongoose = require("mongoose");
const router = express.Router();

const Deposit = require("../models/deposit");
const UserWallet = require("../models/userWallet");
const User = require("../models/users");

const propagateDepositAndEvaluateRanks =
  require("../utils/propagateDepositAndEvaluateRanks");

const distributeReferralBonus =
  require("../utils/distributeReferralBonus");

const verifyNowPaymentsSignature =
  require("../utils/verifyNowPaymentsSignature");

router.post("/nowpayments", async (req, res) => {

  if (!verifyNowPaymentsSignature(req)) {
    console.error("❌ Invalid NOWPayments signature");
    return res.status(401).send("Invalid signature");
  }

  const payload = JSON.parse(req.body.toString("utf8"));


  const { payment_status, payment_id, actually_paid, order_id, pay_address } = payload;


  const session = await mongoose.startSession();
  session.startTransaction();

  try {
    const deposit = await Deposit.findOne({ orderId: order_id }).session(session);
    if (!deposit) {
      await session.abortTransaction();
      return res.status(404).send("Deposit not found");
    }

    if (deposit.status === "completed") {
      await session.commitTransaction();
      return res.status(200).send("Already processed");
    }

    if (payment_status !== "finished") {
      deposit.status = payment_status;
      await deposit.save({ session });
      await session.commitTransaction();
      return res.status(200).send("Status updated");
    }

    // ✅ Mark deposit completed
    deposit.status = "completed";
    deposit.paymentId = payment_id;
    deposit.depositAddress = pay_address;
    await deposit.save({ session });

    // const creditedAmount = deposit.amount;
    const creditedAmount = Number(actually_paid);

    // 💰 Wallet
    const wallet = await UserWallet.findOne({
      userId: deposit.userId,
    }).session(session);

    if (!wallet) throw new Error("Wallet not found");

    wallet.balance += creditedAmount;
    await wallet.save({ session });

    const user = await User.findById(deposit.userId).session(session);
    if (!user) throw new Error("User not found");

    // 🎁 Referral bonus (first deposit only)
    if (user.firstDeposit) {
      await distributeReferralBonus(user, creditedAmount, session);
      user.firstDeposit = false;
      await user.save({ session });
    }

    // 🏆 Rank propagation
    await propagateDepositAndEvaluateRanks(
      deposit.userId,
      creditedAmount,
      session
    );

    await session.commitTransaction();
    return res.status(200).send("Deposit processed");

  } catch (err) {
    await session.abortTransaction();
    console.error("❌ Webhook failure:", err);
    return res.status(500).send("Webhook error");
  } finally {
    session.endSession();
  }
});

module.exports = router;
