const User = require("../models/users.js");
const UserWallet = require("../models/userWallet.js");
const Withdrawal = require("../models/withdrawal.js");

exports.withdrawal = async (req, res) => {
  console.log("withdrawal called");

  try {
    const {
      withdrawalWallet,
      amount,
      paymentMethod,
      paymentAddress,
      userId,
      transactionPin,
      status,
    } = req.body;

    const user = await User.findById(userId);
    const userWallet = await UserWallet.findOne({ userId });

    if (!user) {
      return res.status(404).json({ message: "User not found" });
    }

    if (!userWallet) {
      return res.status(404).json({ message: "User wallet record not found" });
    }

    /* =========================
       TRANSACTION PIN CHECK
    ========================= */
    if (String(user.transactionPin) !== String(transactionPin)) {
      return res.status(400).json({ message: "Incorrect transaction pin" });
    }

    /* =========================
       WALLET VALIDATION
    ========================= */
    const walletType = withdrawalWallet; // e.g. "earnings"

    if (typeof userWallet[walletType] !== "number") {
      return res.status(400).json({
        message: "Invalid withdrawal wallet selected",
      });
    }

    const withdrawalAmount = Number(amount);

    if (withdrawalAmount < 50) {
      return res.status(400).json({
        message: "Minimum withdrawal amount is $50",
      });
    }

    if (withdrawalAmount > userWallet[walletType]) {
      return res.status(400).json({
        message: "Withdrawal amount exceeds available balance",
      });
    }

    /* =========================
       UPDATE BALANCES
    ========================= */
    userWallet[walletType] -= withdrawalAmount;
    userWallet.pendingWithdrawal =
      Number(userWallet.pendingWithdrawal || 0) + withdrawalAmount;

    /* =========================
       CREATE WITHDRAWAL RECORD
    ========================= */
    const newWithdrawal = new Withdrawal({
      withdrawalWallet: walletType,
      amount: withdrawalAmount,
      paymentMethod,
      paymentAddress,
      userId,
      status: status || "pending",
    });

    await newWithdrawal.save();
    await userWallet.save();

    return res.status(201).json({
      message: "Withdrawal created successfully",
      withdrawal: newWithdrawal,
    });
  } catch (error) {
    console.error("Error creating withdrawal:", error);
    return res.status(500).json({
      message: "Failed to create withdrawal",
    });
  }
};



exports.updateWithdrawal = async (req, res, next) => {
  const userId = req.params.id;
  try {
    const approvedWithdrawals = await Withdrawal.find({ userId, status: 'approved' });
    const pendingWithdrawals = await Withdrawal.find({ userId, status: 'pending' });
    console.log("withdrawls", pendingWithdrawals)


    const user = await User.findById(userId);

    if (!user) {
      return res.status(404).json({ message: "User not found" });
    }

    const totalApprovedWithdrawalAmount = approvedWithdrawals.reduce((total, withdrawal) => total + withdrawal.amount, 0);
    const totalPendingWithdrawalAmount = pendingWithdrawals.reduce((total, withdrawal) => total + withdrawal.amount, 0);

    if (user.totalWithdrawal === 0) {
      user.totalWithdrawal = totalApprovedWithdrawalAmount;
    } else if (user.totalWithdrawal > totalApprovedWithdrawalAmount) {
      const adminAdjustedDifference = user.totalWithdrawal - totalApprovedWithdrawalAmount;
      user.totalWithdrawal = totalApprovedWithdrawalAmount + adminAdjustedDifference;
    } else {
      user.totalWithdrawal = totalApprovedWithdrawalAmount;
    }

    if (user.pendingWithdrawal === 0) {
      user.pendingWithdrawal = totalPendingWithdrawalAmount;
    } else {
      user.pendingWithdrawal = totalPendingWithdrawalAmount;
    }

    await user.save();

    return res.status(200).json({
      message: "Withdrawal balances updated successfully",
      totalWithdrawal: user.totalWithdrawal,
      pendingWithdrawal: user.pendingWithdrawal,
    });
  } catch (error) {
    console.error("Error updating total balances:", error);
    return res.status(500).json({ message: "Failed to update withdrawal balances" });
  }
};

exports.getUserWithdrawals = async (req, res, next) => {
  try {
    const userId = req.params.id;

    const userWithdrawals = await Withdrawal.find({ userId });

    if (!userWithdrawals.length) {
      return res.status(404).json({ message: "Withdrawal history not found!" });
    }

    res.status(200).json(userWithdrawals);
  } catch (error) {
    console.error("Error fetching withdrawal:", error);
    res.status(500).json({ message: "Failed to fetch withdrawal" });
  }
};

exports.getAllWithdrawals = async (req, res, next) => {
  try {
    const withdrawals = await Withdrawal.find().populate('userId', 'firstName lastName');
    console.log("All:", withdrawals)

    if (!withdrawals.length) {
      return res.status(404).json({ message: "No withdrawals found" });
    }

    res.status(200).json(withdrawals);
  } catch (error) {
    console.error("Error fetching withdrawals:", error);
    res.status(500).json({ message: "Failed to fetch withdrawals" });
  }
};

exports.getSingleWithdrawal = async (req, res, next) => {
  const withdrawalId = req.params.id;

  try {
    const withdrawal = await Withdrawal.findById(withdrawalId);

    if (!withdrawal) {
      return res.status(404).json({ message: "Withdrawal not found" });
    }

    res.status(200).json({ withdrawal });
  } catch (error) {
    console.error("Error fetching withdrawal:", error);
    res.status(500).json({ message: "Failed to fetch withdrawal" });
  }
};

exports.adminUpdateWithdrawal = async (req, res, next) => {
  try {
    const { amount, paymentMethod, paymentAddress, status } = req.body;

    // Find the existing withdrawal
    const existingWithdrawal = await Withdrawal.findById(req.params.id);

    if (!existingWithdrawal) {
      return res.status(404).json({ message: "Withdrawal not found" });
    }

    // Find the user associated with the withdrawal
    const user = await User.findById(existingWithdrawal.userId);

    if (!user) {
      return res.status(404).json({ message: "User not found" });
    }

    // Check for status change and update user fields
    if (existingWithdrawal.status === 'pending' && status === 'approved') {
      // Move amount from pending to total withdrawal
      user.pendingWithdrawal -= existingWithdrawal.amount;
      user.totalWithdrawal += existingWithdrawal.amount;
    } else if (existingWithdrawal.status === 'approved' && status === 'pending') {
      // Move amount back to pending from total withdrawal
      user.pendingWithdrawal += existingWithdrawal.amount;
      user.totalWithdrawal -= existingWithdrawal.amount;
    }

    // Update the withdrawal status and details
    const updatedWithdrawal = await Withdrawal.findByIdAndUpdate(
      req.params.id,
      { amount, paymentMethod, paymentAddress, status },
      { new: true }
    );

    // Save the updated user information
    await user.save();

    return res.status(200).json(updatedWithdrawal);
  } catch (error) {
    console.error("Error updating withdrawal status:", error);
    return next(error);
  }
};


exports.deleteWithdrawal = async (req, res, next) => {
  const withdrawalId = req.params.id;

  try {
    const withdrawal = await Withdrawal.findByIdAndDelete(withdrawalId);

    if (!withdrawal) {
      return res.status(404).json({ message: "Withdrawal not found" });
    }


    res.status(200).json({ message: "Withdrawal deleted successfully" });
  } catch (error) {
    console.error("Error deleting withdrawal:", error);
    res.status(500).json({ message: "Failed to delete withdrawal" });
  }
};
