const Wallet = require("../models/wallet");

const createWallet = async (req, res, next) => {
  try {
    const { accountName, walletAddress } = req.body;

    // Create a new wallet entry
    const newWallet = new Wallet({
      accountName,
      walletAddress,
    });

    await newWallet.save();

    res.status(201).json({
      message: "Wallet created successfully",
      wallet: newWallet,
    });
  } catch (error) {
    console.error("Error creating wallet:", error);
    res.status(500).json({ message: "Failed to create wallet" });
  }
};

const getAllWallets = async (req, res, next) => {
  try {
    // Fetch all wallets from the database using Mongoose's find method
    const wallets = await Wallet.find();

    // If no wallets are found, return an appropriate message
    if (!wallets || wallets.length === 0) {
      return res.status(404).json({ message: "No wallets found" });
    }

    // Return the found wallets in the response
    res.status(200).json({ wallets });
  } catch (error) {
    // Handle errors that occur during the process
    console.error("Error fetching wallets:", error);
    res.status(500).json({ message: "Failed to fetch wallets" });
  }
};

const getSingleWallet = async (req, res, next) => {
  const walletId = req.params.id;

  try {
    const wallet = await Wallet.findById(walletId);

    if (!wallet) {
      return res.status(404).json({ message: "Wallet not found" });
    }

    res.status(200).json({ wallet });
  } catch (error) {
    console.error("Error fetching wallet:", error);
    res.status(500).json({ message: "Failed to fetch wallet" });
  }
};

const updateWallet = async (req, res, next) => {
  try {
    const { accountName, walletAddress } = req.body;

    const updates = {
      accountName,
      walletAddress,
    };

    const updatedWallet = await Wallet.findByIdAndUpdate(req.params.id, updates, { new: true });

    if (!updatedWallet) {
      return next(errorHandler(404, "Wallet not found"));
    }

    // Return the updated wallet
    res.status(200).json(updatedWallet);
  } catch (error) {
    next(error);
  }
};

module.exports = {
  createWallet,
  getAllWallets,
  getSingleWallet,
  updateWallet,
};
