
const BotPlan = require("../models/botPlan");
const BotSubscription = require("../models/botSubscription");
const UserWallet = require("../models/userWallet");


// POST /api/bots/subscribe
exports.subscribeToBotPlan = async (req, res, next) => {
  try {
    const userId = req.user.id;
    console.log('userId', userId)
    const { planId, amount } = req.body;

    if (!planId || !amount) {
      return res.status(400).json({ message: "Plan and amount are required." });
    }

    // 1️⃣ Fetch plan
    const plan = await BotPlan.findById(planId);
    if (!plan || !plan.isActive) {
      return res.status(400).json({ message: "Invalid or inactive bot plan." });
    }

    // 2️⃣ Fetch user wallet
    const wallet = await UserWallet.findOne({ userId });
    if (!wallet) {
      return res.status(404).json({ message: "Wallet not found." });
    }

    let totalDeduction = 0;

    // 3️⃣ Handle activation fee (ONLY if no free days)
    if (plan.freeDays === 0 && plan.activationFee > 0) {
      if (wallet.balance < plan.activationFee) {
        return res.status(400).json({
          message: "Insufficient balance to pay activation fee.",
        });
      }

      totalDeduction += plan.activationFee;
    }

    // 4️⃣ Validate investment amount AFTER activation fee logic
    if (amount < plan.minAmount || amount > plan.maxAmount) {
      return res.status(400).json({
        message: `Amount must be between ${plan.minAmount} and ${plan.maxAmount}.`,
      });
    }

    totalDeduction += amount;

    // 5️⃣ Final balance check
    if (wallet.balance < totalDeduction) {
      return res.status(400).json({
        message: "Insufficient wallet balance.",
      });
    }

    // 6️⃣ Deduct funds
    wallet.balance -= totalDeduction;
    wallet.activeBalance += amount;
    await wallet.save();

    // 7️⃣ Create subscription
    const subscriptionData = {
      userId,
      botPlanId: plan._id,
      amount,
      dailyRoiPercent: plan.dailyRoiPercent, // snapshot
      freeDaysRemaining: plan.freeDays || 0,
      activationFeePaid: plan.freeDays === 0 && plan.activationFee > 0,
      status: "ACTIVE",
      startedAt: new Date(),
    };

    if (plan.durationDays) {
      subscriptionData.endsAt = new Date(
        Date.now() + plan.durationDays * 24 * 60 * 60 * 1000
      );
    }

    const subscription = await BotSubscription.create(subscriptionData);

    res.status(201).json({
      success: true,
      message:
        plan.freeDays > 0
          ? "Bot plan activated. Free trial started."
          : "Bot plan activated successfully.",
      subscription,
    });
  } catch (err) {
    console.error("Error subscribing to bot plan:", err);
    next(err);
  }
};

// GET /api/bots/my-subscriptions
exports.getMySubscriptions = async (req, res, next) => {
  try {
    const userId = req.user.id;

    const subscriptions = await BotSubscription.find({ userId })
      .populate("botPlanId", "name slug dailyRoiPercent minTradesPerDay maxTradesPerDay")
      .sort({ createdAt: -1 });

    res.status(200).json({
      success: true,
      subscriptions,
    });
  } catch (err) {
    console.error("Error fetching user subscriptions:", err);
    next(err);
  }
};


// GET /api/admin/bot-subscriptions
exports.getAllSubscriptionsAdmin = async (req, res, next) => {
  try {
    const subscriptions = await BotSubscription.find()
      .populate("userId", "firstName lastName email referralId")
      .populate("botPlanId", "name slug dailyRoiPercent freeDays activationFee")
      .sort({ createdAt: -1 });

    res.status(200).json({
      success: true,
      subscriptions,
    });
  } catch (err) {
    console.error("Error fetching all subscriptions:", err);
    next(err);
  }
};


// PUT /api/admin/bot-subscriptions/:id
exports.updateSubscriptionAdmin = async (req, res, next) => {
  try {
    const { id } = req.params;
    const updates = req.body;

    const subscription = await BotSubscription.findById(id);
    if (!subscription) {
      return res.status(404).json({
        message: "Subscription not found.",
      });
    }

    // Allowed fields to update (strict whitelist)
    const allowedUpdates = [
      "status",
      "endsAt",
      "freeDaysRemaining",
    ];

    Object.keys(updates).forEach((key) => {
      if (allowedUpdates.includes(key)) {
        subscription[key] = updates[key];
      }
    });

    await subscription.save();

    res.status(200).json({
      success: true,
      message: "Subscription updated successfully.",
      subscription,
    });
  } catch (err) {
    console.error("Error updating subscription:", err);
    next(err);
  }
};
